﻿using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

using Hims.Shared.UserModels.Common;
using Hims.Shared.UserModels;

namespace Hims.Api.Controllers
{
    using Hims.Api.Models;
    using Hims.Api.Utilities;
    using Hims.Domain.Entities;
    using Hims.Domain.Services;
    using Hims.Shared.DataFilters;
    using Hims.Shared.EntityModels;
    using Hims.Shared.Library.Enums;
    using Microsoft.AspNetCore.Authorization;
    using Shared.UserModels.Filters;

    [Authorize]
    [Route("api/relations")]
    public class RelationsController : BaseController
    {

        /// <summary>
        /// The audit log services.
        /// </summary>
        public readonly IRelationsService relationService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        public RelationsController(IRelationsService service, IAuditLogService auditLogService)
        {
            this.relationService = service;
            this.auditLogServices = auditLogService;

        }




        [AllowAnonymous]
        [HttpPost]
        [Route("fetch")]
        public async Task<ActionResult> FetchAllAsync([FromBody] RelationsModel model)
        {
            model = (RelationsModel)EmptyFilter.Handler(model);
            var response = await this.relationService.FetchAllAsync(model);
            return this.Success(response);

        }

        [AllowAnonymous]
        [HttpPost]
        [Route("edit")]
        public async Task<ActionResult> EditAsync([FromBody] RelationsModel model, [FromHeader] LocationHeader header)
        {
            model = (RelationsModel)EmptyFilter.Handler(model);
            var response = await this.relationService.UpdateAsync(model);


            if (response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.Relations,
                    LogDate = DateTime.Now,
                    LogFrom = short.Parse(model.LoginRoleId.ToString()),
                    LogDescription = $"<b>{model.CreatedByName}</b> has updated Relation of <strong>{model.RelationName}</strong> successfully.",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLog);
            }


            return this.Success(response);
        }



        [HttpPost]
        [Route("modify-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] RelationsModel model, [FromHeader] LocationHeader header)
        {
            model = (RelationsModel)EmptyFilter.Handler(model);
            var response = await this.relationService.ActivateOrDeactivateTest(model);
            if (response == 0)
            {
                return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.Relations,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),

                LogDescription = $" {model.ModifiedByName} has {((bool)model.Active ? "Activated" : "Deactivated")} the Relation <br/>name:<b> '{model.RelationName}'</b> on {DateTime.Now}",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success(response);

        }

    






[AllowAnonymous]
        [HttpPost]
        [Route("add")]

        public async Task<ActionResult> Insert([FromBody] RelationsModel model, [FromHeader] LocationHeader header)
        {

            model = (RelationsModel)EmptyFilter.Handler(model);
            var response = await this.relationService.InsertAsync(model);

            if (response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.Relations,
                    LogDate = DateTime.Now,
                    LogFrom = (short)model.LoginRoleId,
                    LogDescription = $"<b>{model.CreatedByName}</b> has added Relation of <strong>{model.RelationName}</strong> successfully.",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLog);
            }

            return this.Success(response);
        }
    }
}



          
    
